/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/


#include "mx_auto_config.h"
#include "myriexpress.h"
#if MX_OS_WINNT
#include "getopt.h"
#include <winsock2.h>
#else
#include <unistd.h>
#include <netinet/in.h>
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <errno.h>

#include "mx__lib_types.h"
#include "mx__lib.h"
#include "mx__driver_interface.h"
#include "mx__fops.h"

void
usage()
{
  fprintf(stderr, "Usage: mx_counters [args]\n");
  fprintf(stderr, "-b - Board number [0]\n");
  fprintf(stderr, "-c - clear the counters\n");
  fprintf(stderr, "-q - quiet: show only nonzero counters\n");
  fprintf(stderr, "-i - show host interrupt counters\n");
  fprintf(stderr, "-h - help\n");

}

int 
main(int argc, char **argv)
{
  mx_endpt_handle_t fd;
  mx_return_t ret;
  uint32_t counters[1024];
  uint32_t board_id, count;
  uint32_t i; 
  int do_clear, do_quiet, do_irq;
  extern char *optarg;
  int c;
  char *names;
  uint32_t phys_ports;
  mx_irq_counters_t irq;

  board_id = 0;
  do_clear = 0;
  do_irq = 0;
  do_quiet = 0;
  while ((c = getopt(argc, argv, "hqcib:"))!= EOF) switch(c) {
  case 'c':
    do_clear = 1;
    break;
  case 'i':
    do_irq = 1;
    break;
  case 'q':
    do_quiet = 1;
    break;
  case 'b':
     board_id = atoi(optarg);
    break;
  case 'h':
  default:
    usage();
    exit(1);
  }
  
  counters[0] = (uint32_t)board_id;
  mx_init();
  ret = mx_open_any_board(&fd);
  if (ret != MX_SUCCESS) {
    printf("open failed: %s\n", mx_strerror(ret));
    return 0;
  }

  phys_ports = counters[0];
  if (mx__get_num_ports(fd, &phys_ports)) {
    perror("get ports failed");
    phys_ports = 2;
  }
  printf("%d ports\n", phys_ports);

  ret = mx_get_info(NULL, MX_COUNTERS_COUNT, &board_id, sizeof (board_id),
		    &count, sizeof (count));

  if (ret != MX_SUCCESS) {
    perror("get counters count failed");
  }
  names = (char *)malloc (MX_MAX_STR_LEN * count);
  if (names == NULL) {
    printf("out of memory\n");
    goto abort_with_fd;
  }
  ret = mx_get_info(NULL, MX_COUNTERS_LABELS, &board_id, sizeof (board_id),
		     names, MX_MAX_STR_LEN * count);
  if (ret != MX_SUCCESS) {
    perror("get counters strings failed");
    goto abort_with_fd;
  }

  do {
    ret = mx_get_info(NULL, MX_COUNTERS_VALUES, &board_id, sizeof (board_id),
		      counters, sizeof (counters));
    if ((ret != MX_SUCCESS) && (errno != EBUSY)) {
      perror("get counters failed");
      goto abort_with_fd;
    }
  } while (ret);

  for (i = 0; i < count; ++i) {
    if (phys_ports < 2) {
      if (strstr(&names[i * MX_MAX_STR_LEN], "(Port 1)"))
	continue;
    }
    if (do_quiet && !counters[i])
      continue;
    printf("%34s: %10u (0x%x)\n", &names[i * MX_MAX_STR_LEN], 
	   (unsigned int)counters[i], 
           (unsigned int)counters[i]);
  }

  if (do_irq) {
    if (mx__get_irq_counters(fd, board_id, &irq) != 0) {
      perror("get interrupt counters failed -- try enabling MX_DEBUG in the kernel?");
    } else {
      printf("Host irq info: %10u intr, %10u events, %10u intr w/o event\n",
	     irq.count, irq.events, irq.spurious);
    }
  }
  if (do_clear && (mx__clear_counters(fd, board_id) != 0)) {
    perror("clear counters failed");
    exit(33);
  }


 abort_with_fd:
  mx__close(fd);
  mx_finalize();

  return 0;
}
